package gov.va.med.mhv.journal.service.validator;

import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.journal.data.model.FoodJournal;
import gov.va.med.mhv.journal.dto.FoodJournalDTO;
import gov.va.med.mhv.journal.dto.MealItemDTO;
import gov.va.med.mhv.journal.repository.FoodJournalRepository;
import gov.va.med.mhv.journal.web.util.JournalMessagesUtil;

import java.math.BigDecimal;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component 
public class FoodJournalValidator {

	private static final BigDecimal GLASSESOFWATER_MIN_VALUE = new BigDecimal("0");
	private static final BigDecimal GLASSESOFWATER_MAX_VALUE = new BigDecimal("99.99");
	private static final int GLASSESOFWATER_SCALE = 2;
	public static final int COMMENTS_MAX_LENGTH = 255;
	
	@Autowired
	private FoodJournalRepository foodJournalRepository;
	
	@Autowired
	private JournalMessagesUtil messagesUtil;
	
	@Autowired
	private MealItemValidator mealItemValidator;

	/*protected void complexValidationOnComposites() {
		super.complexValidationOnComposites();
	}*/

	public ResponseUtil<FoodJournalDTO> foodJournalValidations(FoodJournalDTO foodJournalDto) {
		ResponseUtil<FoodJournalDTO> response = new ResponseUtil<FoodJournalDTO>();
		response = fieldValidation(foodJournalDto);
		
		// no errors in initial field validations, then perform complex
		// validations
		if (response.getValidationErrors().size() == 0) {
			complexValidation(response, foodJournalDto);
		}
		
		return response;
	}
	
	
	/**
	 * Validates Glasses of Water
	 */
	protected void validateGlassesOfWater(FoodJournalDTO foodJournalDto, LinkedHashMap<String, String> validationErrors) {
		//BigDecimal value = getGlassesOfWater();
		BigDecimal value = foodJournalDto.getGlassesOfWater();
		//check for a value to validate:
		if ( value == null ) {
			//nothing to validate
			return;
		}
		// check min value
		Validations.validateMinValue("Glasses of Water minumum", value, GLASSESOFWATER_MIN_VALUE, "The Glasses of Water ", validationErrors);
		//check max value
		Validations.validateMaxValue("Glasses of Water maximum", value, GLASSESOFWATER_MAX_VALUE, "The Glasses of Water ", validationErrors);
		//check scale
		Validations.validateScale("Glasses of Water scale", value, GLASSESOFWATER_SCALE, "The Glasses of Water ", validationErrors);
	}	
	
	
	/**
	 * Validates createdDate
	 */
	protected void validateCreatedDate(FoodJournalDTO foodJournalDto, LinkedHashMap<String, String> validationErrors) {
		Date value = foodJournalDto.getCreatedDate();
		//check requiredness:		
		if( value == null ) {
			Validations.validateRequired("CreatedDate", null, validationErrors);
						
			//no need to continue:
			return;
		}		
	}	
	
	
	/**
	 * Validates Comments
	 */
	protected void validateComments(FoodJournalDTO foodJournalDto, LinkedHashMap<String, String> validationErrors) {
		String value = StringUtils.trimToNull(foodJournalDto.getComments());
		//check for a value to validate:
		if ( value == null ) {
			//nothing to validate
			return;
		}						
		//check max length
		Validations.validateMaxLength("Comments", value, COMMENTS_MAX_LENGTH, validationErrors);
	}	
	
	
	/**
	 * Validates Date
	 */
	protected void validateJournalDate(FoodJournalDTO foodJournalDto, LinkedHashMap<String, String> validationErrors) {
		Date value = foodJournalDto.getJournalDate();
		//check requiredness:		
		if( value == null ) {
			Validations.validateRequired("JournalDate", null, validationErrors);
						
			//no need to continue:
			return;
		}		
	}	
	
	
	/**
	 * Validates modifiedDate
	 */
	protected void validateModifiedDate(FoodJournalDTO foodJournalDto, LinkedHashMap<String, String> validationErrors) {
		Date value = foodJournalDto.getModifiedDate();
		//check for a value to validate:
		if ( value == null ) {
			//nothing to validate
			return;
		}						
	}	
	
	/**
	 * Validates UserProfile
	 */
	protected void validateUserProfile(FoodJournalDTO foodJournalDto, LinkedHashMap<String, String> validationErrors) {

		if (foodJournalDto.getUserId() == null) {
			//addError("userProfile", "null.not.allowed", new Object[] { "userProfile" });
			validationErrors.put("UserId", String.format(messagesUtil.getNullNotAllowed(), "UserProfileId"));
			// no need to continue:
			return;
		}

		/*if (getUserProfile() == null) {
			addError("userProfile", "invalid.reference", new Object[] { "userProfile" });
		}*/
	}

	
	/**
	 * Executes all field-level validations
	 */
	protected ResponseUtil fieldValidation(FoodJournalDTO foodJournalDto) {
		ResponseUtil response = new ResponseUtil();
		LinkedHashMap<String, String> validationErrors = new LinkedHashMap<String, String>();

		validateGlassesOfWater(foodJournalDto, validationErrors);
		validateCreatedDate(foodJournalDto, validationErrors);
		validateComments(foodJournalDto, validationErrors);
		validateJournalDate(foodJournalDto, validationErrors);
		validateModifiedDate(foodJournalDto, validationErrors);

		//call field validation on children:
		if (null != foodJournalDto.getMealItems() && foodJournalDto.getMealItems().size() > 0) {
			for (MealItemDTO mealItemDto : foodJournalDto.getMealItems()) {
				ResponseUtil mealItemValidationResponse = mealItemValidator.fieldValidation(mealItemDto);
				if (null != mealItemValidationResponse.getValidationErrors() 
						&& mealItemValidationResponse.getValidationErrors().size() > 0) {
					validationErrors.putAll(mealItemValidationResponse.getValidationErrors());
				}
			}
		}
		
		
		validateInformationEntered(foodJournalDto, validationErrors);
		
		if (foodJournalDto.getComments() == null && foodJournalDto.getComments().trim().length() == 0) {
			
			if (foodJournalDto.getItem().isEmpty() && foodJournalDto.getMealItems().size() == 0 ) {
				validationErrors.put("Food/Beverage Item is a required field", "Food/Beverage Item is a required field");
			}
		
			if (foodJournalDto.getMealType().isEmpty() && foodJournalDto.getMealItems().size() == 0 ) {
				validationErrors.put("Meal Type is a required field", "Meal Type is a required field");
			}
		}	
		
		response.setValidationErrors(validationErrors);
		
		return response;
	}
	
	private void validateInformationEntered(FoodJournalDTO foodJournalDto, LinkedHashMap<String, String> validationErrors) {
		if (foodJournalDto.getGlassesOfWater().intValue() == 0 && StringUtils.trimToNull(foodJournalDto.getComments()) == null 
				&& (foodJournalDto.getMealItems() == null || (null != foodJournalDto.getMealItems() && foodJournalDto.getMealItems().size() ==0))) {
			validationErrors.put("sei.fj.noInformation", messagesUtil.getFjNoInformation());
		}
	}

	protected void complexValidation(ResponseUtil response, FoodJournalDTO foodJournalDto) {
		LinkedHashMap<String, String> validationErrors = response.getValidationErrors();
		
		validateSingleEntryForDate(foodJournalDto, validationErrors);
		validateUserProfile(foodJournalDto, validationErrors);
		
		response.setValidationErrors(validationErrors);
	}
	
	/**
	 * Ensure that only one food journal exists for a
	 * particular date
	 *
	 */
	private void validateSingleEntryForDate(FoodJournalDTO foodJournalDto, LinkedHashMap<String, String> validationErrors) {
		if (foodJournalDto.getFoodJournalId() == null && null != foodJournalDto.getJournalDate()) {
			List<FoodJournal> journals = foodJournalRepository.findFoodJournalsForUserAndDate(foodJournalDto.getUserId(), foodJournalDto.getJournalDate());
			
			if (journals.size() > 0) {
				validationErrors.put("singleEntryForDate", messagesUtil.getSeiFJDateAlreadyExists());
				/*if (getFoodJournalId() == null) {
					addError("sei.fj.dateAlreadyExists",
							 new String [] {"journalDate"},
							 new Object[] { getJournalDate().toString() });
				}
				else {
					FoodJournalBO existingJournal = (FoodJournalBO) journals.get(0);
					
					if (!getFoodJournalId().equals(existingJournal.getFoodJournalId())) {
						addError("sei.fj.dateAlreadyExists",
								 new String [] {"journalDate"},
								 new Object[] { getJournalDate().toString() });
					}
				}*/
			}
		}
	}
}